/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * \file    DM388_EVM.c
 *
 * \brief   Generic routines for DM388 EVM
 *
 * This file contains the generic routines for test codes for EVM.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "dm388_types.h"
#include "dm388_EVM.h"
#include "version.h"

#include <stdio.h>

INT32 platform_write(const char *fmt, ... );
int CLKIN =    20;
int HSMODE,CLKOUT = 0;
unsigned int gu8AdpllClkInit = 0;
unsigned int DDR_SIZE;

 /*
 *====================
 * Function implementations
 *====================
 */

void init_delay(void)
{
    int del_cnt;
    for(del_cnt=0;del_cnt<200;del_cnt++);
}

void wait_delay(UWORD32 noopcount)
{
  int i;
  for(i=0;i<noopcount;i++)
  {
  }
}

void ControlModule_ClkEnable(void)
{
    platform_write("\tPRCM for Control Module in Progress \n");
    /*Enable the  Clocks*/
    WR_MEM_32(CM_CTRL_CLKCTRL,   2);
    while(RD_MEM_32(CM_CTRL_CLKCTRL)!=0x2);
    platform_write("\tPRCM for Control Module Done \n");
}

void Unlock_PLL_Control_MMR(void)
{
    *(unsigned int *)0x481C5040 = 0x1EDA4C3D;
    *(unsigned int *)0x48140060 = 0x2FF1AC2B;
    *(unsigned int *)0x48140064 = 0xF757FDC0;
    *(unsigned int *)0x48140068 = 0xE2BC3A6D;
    *(unsigned int *)0x4814006c = 0x1EBF131D;
    *(unsigned int *)0x48140070 = 0x6F361E05;
    platform_write("\n PLL and Control MMR unlock done ... \n");
}

void OpenFireWall(void)
{
  if( (RD_MEM_32(CONTROL_STATUS) & 0x700) ==0)
  {
    platform_write("\tDevice type is TEST \n","Output");
    platform_write("\tOpen the Firewall for public \n","Output");
    WR_MEM_32(GPMC_FW        ,    DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(OCMC_RAM0_FW,        DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(DUCATI_FW,        DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(PCIE_FW,            DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(IVAHD_SL2_FW,        DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(IVAHD_CFG_FW,        DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(MCASP_FW,         DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(SecuritySS_FW,     DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(HDMI_FW,             DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(TPTC_FW,             DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(TPCC_FW,             DM385_TEST_DEVICE_FW_VAL);
    WR_MEM_32(USB_FW,             DM385_TEST_DEVICE_FW_VAL);
  }
  else if( (RD_MEM_32(CONTROL_STATUS) & 0x700) ==0x300)
  {
    platform_write("\tDevice type is GP \n","Output");
  }
  platform_write("\t OpenFirewall completed \n","Output");
}

void ALL_ADPLL_CLOCKS_ENABLE_API(void)
{
    if (((RD_MEM_32(DSS_PLL_BASE + STATUS)) & 0x00000600) == 0x00000600)
       gu8AdpllClkInit = 1;
    else
       gu8AdpllClkInit = 0;

   if (gu8AdpllClkInit == 0)
   {
        //platform_write("\t ****  DM385 ALL ADPLL INIT IS In Progress ......... \n","Output",1,1,1);
        PLL_SETUP();
        EMACClkEnable();
        gu8AdpllClkInit = 1;
        //platform_write("\t ****  DM385 ALL ADPLL INIT IS  Done ************** \n","Output",1,1,1);
  }
}

void EMACClkEnable(void)
{
    platform_write("\tCONFIGURE PRCM CLOCKS of EMAC in progress \n\r");

    WR_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL,    0x2); /* Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    while(RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL)!=0x302);        /*Poll for Module is functional*/

    platform_write("\tPRCM CLOCKS of EMAC  is complete \n\r");
}

void PLL_SETUP(void)
{
    //pll_name (CLKINP,N , M, M2);
    cmdMPUPLL(CLKIN,1, 60 ,1);
    cmdL3PLL(CLKIN,19,800,4);
    cmdDSSPLL(CLKIN,19, 800, 4);
    cmdISSPLL(CLKIN,19, 800 ,4);
    cmdIVAPLL(CLKIN,19, 532, 2);
    cmdUSBPLL(CLKIN,19,960,5);
    cmdVIDEO0PLL(CLKIN,19, 1000,4);
    cmdVIDEO1PLL(CLKIN,19, 594,4);
    cmdHDMIPLL(CLKIN,19, 1485,10);
    cmdDDRPLL(CLKIN,19,800, 2);
    cmdAUDIOPLL(CLKIN,19,800,4);
	SATA_PLL_Config();
}

void cmdMPUPLL(int CLKIN,int N, int M, int M2)
{
    SABERTOOTH_PLL_Config(CLKIN,N,M,M2);
    //platform_write("\t SABERTOOTH ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
}

void SABERTOOTH_PLL_Config(UWORD32 CLKIN,UWORD32 N,UWORD32 M,UWORD32 M2)
{
    UWORD32 rval_ctrl,ref_clk,clk_out = 0;
    UWORD32 m2nval,mn2val = 0;
    ref_clk     = CLKIN/(N+1);
    clk_out     = (ref_clk*M)/M2;

    m2nval = (M2<<16) | N; //need to check the bit position of M2
    mn2val =  M;
    WR_MEM_32(SABERTOOTHPLL_M2NDIV      ,m2nval);
    WR_MEM_32(SABERTOOTHPLL_MN2DIV      ,mn2val);
    WR_MEM_32(SABERTOOTHPLL_TENABLEDIV  ,0x1);
    wait_delay(3);
    WR_MEM_32(SABERTOOTHPLL_TENABLEDIV  ,0x0);
    wait_delay(3);
    WR_MEM_32(SABERTOOTHPLL_TENABLE     ,0x1);
    wait_delay(3);
    WR_MEM_32(SABERTOOTHPLL_TENABLE     ,0x0);
    wait_delay(3);
    rval_ctrl = RD_MEM_32(SABERTOOTHPLL_CLKCTRL);
    WR_MEM_32(SABERTOOTHPLL_CLKCTRL,(rval_ctrl & 0xff7fffff) | 0x1);
    while (( (RD_MEM_32(SABERTOOTHPLL_STATUS)) & 0x00000600) != 0x00000600);
    wait_delay(10);
    CLKOUT = clk_out;

}

void cmdL3PLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2)
        {
        PLL_Clocks_Config(L3_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
        //platform_write("\t L3  ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
        }
    else if (HSMODE == 1)
        {
        PLL_Clocks_Config(L3_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
        //platform_write("\t L3  ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
        }
    else
        {
        //platform_write("\t L3 PLL NOT Configured.Wrong DCOCLK Output\n");
        }
}


void cmdDSSPLL(int CLKIN, int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(DSS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
           //platform_write("\t DSS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(DSS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
           //platform_write("\t DSS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t DSS PLL NOT Configured.Wrong DCOCLK Output\n");
    }

}

void cmdIVAPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(IVA_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
         //platform_write("\t IVA ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(IVA_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
         //platform_write("\t IVA ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t IVA PLL NOT Configured.Wrong DCOCLK Output\n");
    }

}

void cmdISSPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(ISS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
         //platform_write("\t ISS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(ISS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
         //platform_write("\t ISS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t ISS PLL NOT Configured.Wrong DCOCLK Output\n");
    }

}

void cmdUSBPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(USB_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2+ADPLLJ_CLKCRTL_CLKDCO);
         //platform_write("\t USB ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(USB_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1+ADPLLJ_CLKCRTL_CLKDCO);
         //platform_write("\t USB ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t USB PLL NOT Configured.Wrong DCOCLK Output\n");
    }

}

void cmdVIDEO0PLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(VIDEO_0_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
        //platform_write("\t VIDEO-0 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(VIDEO_0_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
        //platform_write("\t VIDEO-0 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
        //platform_write("\t VIDEO-0 PLL NOT Configured.Wrong DCOCLK Output\n");
    }
}

void cmdVIDEO1PLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(VIDEO_1_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
        //platform_write("\t VIDEO-1 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(VIDEO_1_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
        //platform_write("\t VIDEO-1 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
        //platform_write("\t VIDEO-1 PLL NOT Configured.Wrong DCOCLK Output\n");
    }
}

void cmdHDMIPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(HDMI_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2+ADPLLJ_CLKCRTL_CLKDCO);
         //platform_write("\t VIDEO-2/HDMI ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(HDMI_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1+ADPLLJ_CLKCRTL_CLKDCO);
         //platform_write("\t VIDEO-2/HDMI ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t VIDEO-2/HDMI PLL NOT Configured.Wrong DCOCLK Output\n");
    }
}

void cmdDDRPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(DDR_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
         //platform_write("\t DDR ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(DDR_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
         //platform_write("\t DDR ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t DDR PLL NOT Configured.Wrong DCOCLK Output\n");
    }
}

void cmdAUDIOPLL(int CLKIN,int N, int M, int M2)
{
    DCOCLK_COMP(CLKIN,N,M);
    if(HSMODE == 2){
        PLL_Clocks_Config(AUDIO_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
         //platform_write("\t AUDIO ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else if (HSMODE == 1){
        PLL_Clocks_Config(AUDIO_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
         //platform_write("\t AUDIO ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
    }
    else {
              //platform_write("\t AUDIO PLL NOT Configured.Wrong DCOCLK Output\n");
    }

}

void SATA_PLL_Config(void)
{
	WR_MEM_32(SATA0_PLLCFG1, 0xC12C003C);	//cfgpll1
	wait_delay(2000);
	WR_MEM_32(SATA0_PLLCFG3, 0x004008E0);	//cfgpll3
	wait_delay(2000);
	//wait for bias to be stable --50us
	WR_MEM_32(SATA0_PLLCFG0, 0x80000004);	//cfgpll0
	wait_delay(2000);
	WR_MEM_32(SATA0_PLLCFG0, 0x80000014);	//cfgpll0
	wait_delay(2000);
	WR_MEM_32(SATA0_PLLCFG0, 0x80000016);	//cfgpll0
	wait_delay(2000);
	WR_MEM_32(SATA0_PLLCFG0, 0xc0000017);	//cfgpll0
	wait_delay(2000);

	//poll the status field to check if pll lock occured.
	while( ((*(unsigned int *)0x48140734) & 0x1) != 0x1)
	{
	}
	//platform_write("\t --- SATA PLL Locked\n","Output",1,1,1);
}

void DDR3_Initialization(void)
{
  if (DDR_FREQ == 800)
    DDR3_Init_400MHz_Config();
  else if (DDR_FREQ == 1066)
    DDR3_Init_533MHz_Config();

}

void  DDR3_Init_400MHz_Config(void)
{
    ALL_ADPLL_CLOCKS_ENABLE_API();
    DDR_SIZE=ONE_GB;
    platform_write("\t ****  Configuring DDR PLL to 400 MHz......... \n");
    cmdDDRPLL(CLKIN,19,800, 2);
    platform_write("\tDM385 DDR3 EVM EMIF0 configuration in progress......... \n");
    cmd_DDR3_EMIF0_Config(DDR3_PHY_RD_DQS_CS0_DEFINE,DDR3_PHY_WR_DQS_CS0_DEFINE,DDR3_PHY_RD_DQS_GATE_CS0_DEFINE,DDR3_PHY_WR_DATA_CS0_DEFINE,DDR3_EMIF_DDRPHYCR_DEFINE_400,DDR3_EMIF_TIM1_DEFINE_400,DDR3_EMIF_TIM2_DEFINE_400,DDR3_EMIF_TIM3_DEFINE_400,DDR3_EMIF_REF_CTRL_DEFINE2_400,DDR3_EMIF_SDRAM_CONFIG_DEFINE_400);
    platform_write("\tDM385 DDR3 EVM EMIF0 configuration is DONE. \n");
}


void  DDR3_Init_533MHz_Config(void)
{
    ALL_ADPLL_CLOCKS_ENABLE_API();
    DDR_SIZE=ONE_GB;
    platform_write("\t ****  Configuring DDR PLL to 533 MHz......... \n");
    cmdDDRPLL(CLKIN,19,1066, 2);
    platform_write("\tDM385 DDR3 EVM EMIF0 configuration in progress......... \n");
    cmd_DDR3_EMIF0_Config(DDR3_PHY_RD_DQS_CS0_DEFINE,DDR3_PHY_WR_DQS_CS0_DEFINE,DDR3_PHY_RD_DQS_GATE_CS0_DEFINE,DDR3_PHY_WR_DATA_CS0_DEFINE,DDR3_EMIF_DDRPHYCR_DEFINE_533,DDR3_EMIF_TIM1_DEFINE_533,DDR3_EMIF_TIM2_DEFINE_533,DDR3_EMIF_TIM3_DEFINE_533,DDR3_EMIF_REF_CTRL_DEFINE2_533,DDR3_EMIF_SDRAM_CONFIG_DEFINE_533);
    platform_write("\tDM385 DDR3 EVM EMIF0 configuration is DONE. \n");
}

void cmd_DDR3_EMIF0_Config(UWORD32 ddr3_phy_rd_dqs_cs0_arg,UWORD32 ddr3_phy_wr_dqs_cs0_arg,UWORD32 ddr3_phy_rd_dqs_gate_cs0_arg,UWORD32 ddr3_phy_wr_data_cs0_arg,UWORD32 ddr3_emif_read_latency_arg,UWORD32 ddr3_emif_tim1_arg,UWORD32 ddr3_emif_tim2_arg,UWORD32 ddr3_emif_tim3_arg,UWORD32 ddr3_emif_ref_ctrl_arg,UWORD32 ddr3_emif_sdram_config_arg)
{
    Emif_PRCM_Clk_Enable();
    platform_write("\tDM385 DDR,DMM PRCM configuration is Done \n");

    Cmd_Macro_Config(DDR_PHY0,DDR3_PHY_INVERT_CLKOUT_OFF,DDR3_PHY_CTRL_SLAVE_RATIO_CS0_DEFINE,PHY_CMD0_DLL_LOCK_DIFF_DEFINE);
    Data_Macro_Config(DATA_MACRO_0,DDR_PHY0,ddr3_phy_rd_dqs_cs0_arg,ddr3_phy_wr_dqs_cs0_arg,ddr3_phy_rd_dqs_gate_cs0_arg,ddr3_phy_wr_data_cs0_arg);
    Data_Macro_Config(DATA_MACRO_1,DDR_PHY0,ddr3_phy_rd_dqs_cs0_arg,ddr3_phy_wr_dqs_cs0_arg,ddr3_phy_rd_dqs_gate_cs0_arg,ddr3_phy_wr_data_cs0_arg);
    Data_Macro_Config(DATA_MACRO_2,DDR_PHY0,ddr3_phy_rd_dqs_cs0_arg,ddr3_phy_wr_dqs_cs0_arg,ddr3_phy_rd_dqs_gate_cs0_arg,ddr3_phy_wr_data_cs0_arg);
    Data_Macro_Config(DATA_MACRO_3,DDR_PHY0,ddr3_phy_rd_dqs_cs0_arg,ddr3_phy_wr_dqs_cs0_arg,ddr3_phy_rd_dqs_gate_cs0_arg,ddr3_phy_wr_data_cs0_arg);

    platform_write("\tDM385 DDR PHY Configuration is Done \n");

    WR_MEM_32(DDR0_IO_CTRL,0x00030303);
    platform_write("\tDM385 DDR IO Control Configuration is Done \n");

    Vtp_Enable();
    platform_write("\tDM385 VTP Configuration is Done \n");
    if(DDR_SIZE==TWO_GB)
    {
        /*Program the DMM to Access EMIF0*/
        WR_MEM_32(DMM_LISA_MAP__0, 0x80700100);
        WR_MEM_32(DMM_LISA_MAP__1, 0x80700100);
        WR_MEM_32(DMM_LISA_MAP__2, 0x80700100);
        WR_MEM_32(DMM_LISA_MAP__3, 0x80700100);

        while(RD_MEM_32(DMM_LISA_MAP__0)!=0x80700100);
        while(RD_MEM_32(DMM_LISA_MAP__1)!=0x80700100);
        while(RD_MEM_32(DMM_LISA_MAP__2)!=0x80700100);
        while(RD_MEM_32(DMM_LISA_MAP__3)!=0x80700100);

        WR_MEM_32(DMM_PAT_BASE_ADDR, 0x80000000);
    }

    else if(DDR_SIZE==ONE_GB)
    {
        /*Program the DMM to Access EMIF0*/
        WR_MEM_32(DMM_LISA_MAP__0, 0x80600100);
        WR_MEM_32(DMM_LISA_MAP__1, 0x80600100);
        WR_MEM_32(DMM_LISA_MAP__2, 0x80600100);
        WR_MEM_32(DMM_LISA_MAP__3, 0x80600100);

        while(RD_MEM_32(DMM_LISA_MAP__0)!=0x80600100);
        while(RD_MEM_32(DMM_LISA_MAP__1)!=0x80600100);
        while(RD_MEM_32(DMM_LISA_MAP__2)!=0x80600100);
        while(RD_MEM_32(DMM_LISA_MAP__3)!=0x80600100);

        WR_MEM_32(DMM_PAT_BASE_ADDR, 0x80000000);
    }

    platform_write("\tDM385 DMM LISA register Configuration is done for %d GBytes \n",DDR_SIZE);

    Emif0_MMR_Config(ddr3_emif_read_latency_arg,ddr3_emif_tim1_arg,ddr3_emif_tim2_arg,ddr3_emif_tim3_arg,ddr3_emif_ref_ctrl_arg,ddr3_emif_sdram_config_arg);
}

void Emif_PRCM_Clk_Enable(void)
{
    WR_MEM_32(CM_DEFAULT_FW_CLKCTRL, 0x2);        /*Enable the EMIF FireWall Clocks*/
    WR_MEM_32(CM_DEFAULT_L3_FAST_CLKSTCTRL, 0x2); /*Enable the Power Domain Transition of L3 Fast Domain Peripheral*/
    WR_MEM_32(CM_DEFAULT_EMIF_0_CLKCTRL,    0x2); /*Enable EMIF0 Clock*/
    WR_MEM_32(CM_DEFAULT_DMM_CLKCTRL,       0x2); /*Enable DMM Clock*/
    while((RD_MEM_32(CM_DEFAULT_L3_FAST_CLKSTCTRL) & 0x300)!=0x300);    /*Poll for L3_FAST_GCLK  & DDR_GCLK  are active*/
    while(RD_MEM_32(CM_DEFAULT_EMIF_0_CLKCTRL)!=0x2);       /*Poll for Module is functional*/
    while(RD_MEM_32(CM_DEFAULT_DMM_CLKCTRL)!=0x2);          /*Poll for Module is functional*/
}

void Cmd_Macro_Config(UWORD32 ddr_phy_num,UWORD32 invert_clk_out,UWORD32 ctrl_slave_ratio_cs0,UWORD32 cmd_dll_lock_diff)
{
    WR_MEM_32(CMD0_REG_PHY0_INVERT_CLKOUT_0, invert_clk_out);
    WR_MEM_32(CMD1_REG_PHY0_INVERT_CLKOUT_0, invert_clk_out);
    WR_MEM_32(CMD2_REG_PHY0_INVERT_CLKOUT_0, invert_clk_out);

    WR_MEM_32(CMD0_REG_PHY0_CTRL_SLAVE_RATIO_0,(ctrl_slave_ratio_cs0 << 10 | ctrl_slave_ratio_cs0));
    WR_MEM_32(CMD1_REG_PHY0_CTRL_SLAVE_RATIO_0,(ctrl_slave_ratio_cs0 << 10 | ctrl_slave_ratio_cs0));
    WR_MEM_32(CMD2_REG_PHY0_CTRL_SLAVE_RATIO_0,(ctrl_slave_ratio_cs0 << 10 | ctrl_slave_ratio_cs0));

    WR_MEM_32(CMD0_REG_PHY0_DLL_LOCK_DIFF_0,cmd_dll_lock_diff);
    WR_MEM_32(CMD1_REG_PHY0_DLL_LOCK_DIFF_0,cmd_dll_lock_diff);
    WR_MEM_32(CMD2_REG_PHY0_DLL_LOCK_DIFF_0,cmd_dll_lock_diff);
}

void Data_Macro_Config(UWORD32 dataMacroNum,UWORD32 ddr_phy_num,UWORD32 rd_dqs_cs0,UWORD32 wr_dqs_cs0,UWORD32 rd_dqs_gate_cs0,UWORD32 wr_data_cs0)
{
    UWORD32 BaseAddrOffset;
    if(dataMacroNum == DATA_MACRO_0)
        BaseAddrOffset = 0x00;
    else if(dataMacroNum == DATA_MACRO_1)
        BaseAddrOffset = 0xA4;
    else if(dataMacroNum == DATA_MACRO_2)
        BaseAddrOffset = 0x148;
    else if(dataMacroNum == DATA_MACRO_3)
        BaseAddrOffset = 0x1EC;

    WR_MEM_32((DATA0_REG_PHY0_RD_DQS_SLAVE_RATIO_0 + BaseAddrOffset), (rd_dqs_cs0 << 10 | rd_dqs_cs0));
    WR_MEM_32((DATA0_REG_PHY0_WR_DQS_SLAVE_RATIO_0 + BaseAddrOffset), (wr_dqs_cs0 << 10 | wr_dqs_cs0));
    WR_MEM_32((DATA0_REG_PHY0_WRLVL_INIT_RATIO_0 + BaseAddrOffset),   (PHY_WRLVL_INIT_CS1_DEFINE << 10  | PHY_WRLVL_INIT_CS0_DEFINE));
    WR_MEM_32((DATA0_REG_PHY0_GATELVL_INIT_RATIO_0 + BaseAddrOffset), (PHY_GATELVL_INIT_CS1_DEFINE << 10 | PHY_GATELVL_INIT_CS0_DEFINE));
    WR_MEM_32((DATA0_REG_PHY0_RD_DQS_GATE_SLAVE_RATIO_0 + BaseAddrOffset),(rd_dqs_gate_cs0 << 10  | rd_dqs_gate_cs0));
    WR_MEM_32((DATA0_REG_PHY0_WR_DATA_SLAVE_RATIO_0 + BaseAddrOffset),(wr_data_cs0 << 10 | wr_data_cs0));
    //-WR_MEM_32((DATA0_REG_PHY0_USE_RANK0_DELAYS + BaseAddrOffset),     PHY_REG_USE_RANK0_DELAY_DEFINE);//- default is 0; for mDDR need to set as 1
    WR_MEM_32((DATA0_REG_PHY0_DLL_LOCK_DIFF_0 + BaseAddrOffset),      PHY_DLL_LOCK_DIFF_DEFINE);
}

void Vtp_Enable(void)
{
    // Write 1 to ENABLE bit
    WR_MEM_32(VTP0_CTRL_REG, RD_MEM_32(VTP0_CTRL_REG) | 0x00000040 );

    // Write 0 to CLRZ bit
    WR_MEM_32(VTP0_CTRL_REG, RD_MEM_32(VTP0_CTRL_REG) & 0xfffffffe );

    // Write 1 to CLRZ bit
    WR_MEM_32(VTP0_CTRL_REG, RD_MEM_32(VTP0_CTRL_REG) | 0x00000001 );

    // Read VTP control registers & check READY bits
    while( (RD_MEM_32(VTP0_CTRL_REG) & 0x00000020) != 0x20);
}

void Emif0_MMR_Config(UWORD32 read_latency,UWORD32 tim1,UWORD32 tim2,UWORD32 tim3,UWORD32 ref_ctrl,UWORD32 sdram_config)
{
    /*Program EMIF0 CFG Registers*/
    WR_MEM_32(EMIF4_0_DDR_PHY_CTRL_1, read_latency);
    WR_MEM_32(EMIF4_0_DDR_PHY_CTRL_1_SHADOW, read_latency);

    WR_MEM_32(EMIF4_0_SDRAM_TIM_1, tim1);
    WR_MEM_32(EMIF4_0_SDRAM_TIM_1_SHADOW, tim1);

    WR_MEM_32(EMIF4_0_SDRAM_TIM_2, tim2);
    WR_MEM_32(EMIF4_0_SDRAM_TIM_2_SHADOW, tim2);

    WR_MEM_32(EMIF4_0_SDRAM_TIM_3, tim3);
    WR_MEM_32(EMIF4_0_SDRAM_TIM_3_SHADOW, tim3);

    WR_MEM_32(EMIF4_0_SDRAM_CONFIG, sdram_config);

    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL, 0x10000000|DDR3_EMIF_REF_CTRL_DEFINE1);
    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL_SHADOW, DDR3_EMIF_REF_CTRL_DEFINE1);

    WR_MEM_32(EMIF4_0_SDRAM_ZQCR,DDR3_EMIF_SDRAM_ZQCR_DEFINE);

    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL, DDR3_EMIF_REF_CTRL_DEFINE1);
    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL_SHADOW, DDR3_EMIF_REF_CTRL_DEFINE1);

    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL, ref_ctrl);
    WR_MEM_32(EMIF4_0_SDRAM_REF_CTRL_SHADOW, ref_ctrl);
}

void DCOCLK_COMP(int CLKIN,int N, int M)
 {
         int DCOCLK;
        DCOCLK = (CLKIN/(N+1))*M;

        if(DCOCLK >= 500 && DCOCLK < 1000){
                HSMODE = 2;  //HS2 Mode
        }
        else if(DCOCLK >= 1000 && DCOCLK < 2000){
                HSMODE = 1;  //HS1 Mode
        }
        else HSMODE = 0;  //wrong configuration


 }

void PLL_Clocks_Config(UWORD32 Base_Address,UWORD32 CLKIN,UWORD32 N,UWORD32 M,UWORD32 M2,UWORD32 CLKCTRL_VAL)
{
    UWORD32 m2nval,mn2val,read_clkctrl,clk_out,ref_clk,clkout_dco = 0;
    m2nval = (M2<<16) | N;
    mn2val =  M;
    ref_clk     = CLKIN/(N+1);
    clkout_dco  = ref_clk*M;
    clk_out     = clkout_dco/M2;
    WR_MEM_32(Base_Address+CLKCTRL, RD_MEM_32(Base_Address+CLKCTRL)|0x00800000);
    while (( (RD_MEM_32(Base_Address+STATUS)) & 0x00000101) != 0x00000101);
    WR_MEM_32(Base_Address+CLKCTRL, RD_MEM_32(Base_Address+CLKCTRL)& 0xfffffffe);
    wait_delay(3);
    WR_MEM_32((Base_Address+M2NDIV    ),m2nval);
    WR_MEM_32((Base_Address+MN2DIV    ),mn2val);
    wait_delay(3);
    WR_MEM_32((Base_Address+TENABLEDIV),0x1);
    wait_delay(3);
    WR_MEM_32((Base_Address+TENABLEDIV),0x0);
    wait_delay(3);
    WR_MEM_32((Base_Address+TENABLE   ),0x1);
    wait_delay(3);
    WR_MEM_32((Base_Address+TENABLE   ),0x0);
    wait_delay(3);
    read_clkctrl = RD_MEM_32(Base_Address+CLKCTRL);
    //configure the TINITZ(bit0) and CLKDCO BITS IF REQUIRED
    WR_MEM_32(Base_Address+CLKCTRL,(read_clkctrl & 0xff7fe3ff) | CLKCTRL_VAL);
    read_clkctrl = RD_MEM_32(Base_Address+CLKCTRL);
    // poll for the freq,phase lock to occur
    while (( (RD_MEM_32(Base_Address+STATUS)) & 0x00000600) != 0x00000600);
    //wait fot the clocks to get stabized
    wait_delay(10);
    CLKOUT    = clk_out;
}

void GPMCClkEnable(void)
{
    platform_write("\tEnabling Clock for GPMC is in Progress, Please wait.....  \n","Output",1,1,1);
    WR_MEM_32(CM_ALWON_GPMC_CLKCTRL,    0x2); /*Enable GPMC Clock*/
    while(RD_MEM_32(CM_ALWON_GPMC_CLKCTRL)!=0x2);
    platform_write("\tGPMC Clock is Active\n","Output",1,1,1);
}

void NANDInit(void)
{
   platform_write("\tGPMC PINMUX configuration for NAND in progress, Please wait.....  \n","Output",1,1,1);
   ALL_ADPLL_CLOCKS_ENABLE_API();
   GPMCClkEnable();


   WR_MEM_32(PINCNTL235,    RD_MEM_32(PINCNTL235)  | 0x1);
   WR_MEM_32(PINCNTL243,    RD_MEM_32(PINCNTL243)  | 0x10);
   WR_MEM_32(PINCNTL244,    RD_MEM_32(PINCNTL244)  | 0x10);
   WR_MEM_32(PINCNTL245,    RD_MEM_32(PINCNTL245)  | 0x10);
   WR_MEM_32(PINCNTL246,    RD_MEM_32(PINCNTL246)  | 0x10);
   WR_MEM_32(PINCNTL247,    RD_MEM_32(PINCNTL247)  | 0x10);
   WR_MEM_32(PINCNTL248,    RD_MEM_32(PINCNTL248)  | 0x10);
   WR_MEM_32(PINCNTL249,    RD_MEM_32(PINCNTL249)  | 0x10);
   WR_MEM_32(PINCNTL250,    RD_MEM_32(PINCNTL250)  | 0x10);
   WR_MEM_32(PINCNTL251,    RD_MEM_32(PINCNTL251)  | 0x10);
   WR_MEM_32(PINCNTL252,    RD_MEM_32(PINCNTL252)  | 0x10);
   WR_MEM_32(PINCNTL253,    RD_MEM_32(PINCNTL253)  | 0x10);
   WR_MEM_32(PINCNTL254,    RD_MEM_32(PINCNTL254)  | 0x10);
   WR_MEM_32(PINCNTL255,    RD_MEM_32(PINCNTL255)  | 0x10);
   WR_MEM_32(PINCNTL256,    RD_MEM_32(PINCNTL256)  | 0x10);
   WR_MEM_32(PINCNTL257,    RD_MEM_32(PINCNTL257)  | 0x10);
   WR_MEM_32(PINCNTL258,    RD_MEM_32(PINCNTL258)  | 0x10);
   WR_MEM_32(PINCNTL89,    RD_MEM_32(PINCNTL89)    | 0x01);
   WR_MEM_32(PINCNTL90,    RD_MEM_32(PINCNTL90)    | 0x01);
   WR_MEM_32(PINCNTL91,    RD_MEM_32(PINCNTL91)    | 0x01);
   WR_MEM_32(PINCNTL92,    RD_MEM_32(PINCNTL92)    | 0x01);
   WR_MEM_32(PINCNTL93,    RD_MEM_32(PINCNTL93)    | 0x01);
   WR_MEM_32(PINCNTL94,    RD_MEM_32(PINCNTL94)    | 0x01);
   WR_MEM_32(PINCNTL95,    RD_MEM_32(PINCNTL95)    | 0x01);
   WR_MEM_32(PINCNTL96,    RD_MEM_32(PINCNTL96)    | 0x01);
   WR_MEM_32(PINCNTL97,    RD_MEM_32(PINCNTL97)    | 0x01);
   WR_MEM_32(PINCNTL98,    RD_MEM_32(PINCNTL98)    | 0x01);
   WR_MEM_32(PINCNTL99,    RD_MEM_32(PINCNTL99)    | 0x01);
   WR_MEM_32(PINCNTL100,   RD_MEM_32(PINCNTL100)   | 0x01);
   WR_MEM_32(PINCNTL101,   RD_MEM_32(PINCNTL101)   | 0x01);
   WR_MEM_32(PINCNTL102,   RD_MEM_32(PINCNTL102)   | 0x01);
   WR_MEM_32(PINCNTL103,   RD_MEM_32(PINCNTL103)   | 0x01);
   WR_MEM_32(PINCNTL104,   RD_MEM_32(PINCNTL104)   | 0x01);
   WR_MEM_32(PINCNTL105,   RD_MEM_32(PINCNTL105)   | 0x01);
   WR_MEM_32(PINCNTL106,    RD_MEM_32(PINCNTL106)  | 0x01);
   WR_MEM_32(PINCNTL107,    RD_MEM_32(PINCNTL107)  | 0x01);
   WR_MEM_32(PINCNTL108,    RD_MEM_32(PINCNTL108)  | 0x01);
   WR_MEM_32(PINCNTL109,    RD_MEM_32(PINCNTL109)  | 0x01);
   WR_MEM_32(PINCNTL110,    RD_MEM_32(PINCNTL110)  | 0x01);
   WR_MEM_32(PINCNTL111,    RD_MEM_32(PINCNTL111)  | 0x01);
   WR_MEM_32(PINCNTL112,    RD_MEM_32(PINCNTL112)  | 0x01);
   WR_MEM_32(PINCNTL122,    RD_MEM_32(PINCNTL122)  | 0x01);
   WR_MEM_32(PINCNTL123,    RD_MEM_32(PINCNTL123)  | 0x01);
   WR_MEM_32(PINCNTL124,    RD_MEM_32(PINCNTL124)  | 0x01);
   WR_MEM_32(PINCNTL125,    RD_MEM_32(PINCNTL125)  | 0x01);
   WR_MEM_32(PINCNTL126,    RD_MEM_32(PINCNTL126)  | 0x01);
   WR_MEM_32(PINCNTL127,    RD_MEM_32(PINCNTL127)  | 0x01);
   WR_MEM_32(PINCNTL128,    RD_MEM_32(PINCNTL128)  | 0x01);
   WR_MEM_32(PINCNTL129,    RD_MEM_32(PINCNTL129)  | 0x01);
   WR_MEM_32(PINCNTL130,    RD_MEM_32(PINCNTL130)  | 0x01);
   WR_MEM_32(PINCNTL131,    RD_MEM_32(PINCNTL131)  | 0x01);
   WR_MEM_32(PINCNTL132,    RD_MEM_32(PINCNTL132)  | 0x01);
   WR_MEM_32(PINCNTL133,    RD_MEM_32(PINCNTL133)  | 0x01);

   WR_MEM_32(0x50000060,    RD_MEM_32(0x50000060) & 0xFFFFFCFF);// Bit 9:8 make 00, Non multiplex
   WR_MEM_32(0x50000078,    RD_MEM_32(0x50000078) & 0xFFFFFFFC);
   WR_MEM_32(0x50000078,    RD_MEM_32(0x50000078) | 2); // Map to 0x02000000 address

   platform_write("\tGPMC initialization complete...\n","Output",1,1,1);
}

void Aic3104Init(void)
{
    platform_write("\t ***** Initializes the DM385 board for AIC3104 test....***** \n");

    ALL_ADPLL_CLOCKS_ENABLE_API();
	I2C2Enable();

    /* DM385 change: Changed to MCA1 instead of MCA2 */
    WR_MEM_32(PINCNTL15, 0x00060004);  /* MCA1- AHCLKX */
    WR_MEM_32(PINCNTL31, 0x00060001);  /* MCA1- ACLKX */
    WR_MEM_32(PINCNTL32, 0x00060001);  /* MCA1- AFSX */
    WR_MEM_32(PINCNTL33, 0x00060001);  /* MCA1-ACLKR */
    WR_MEM_32(PINCNTL34, 0x000C0001);  /* MCA1-AFSR */

    WR_MEM_32(PINCNTL35, 0x00060001);  /* MCA1- AXR0 */
    WR_MEM_32(PINCNTL36, 0x00060001);  /* MCA1- AXR1 */

    MCA1ClkEnable();
    platform_write("\t ***** AIC3104 McASP1 is initialized....***** \n");
}

void I2C0Enable(void)
{
    UARTClkEnable();
    WR_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL,    0x2); /*Enable I2C0 Clock*/
    while(RD_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(PINCNTL263, 0x00060001);
    WR_MEM_32(PINCNTL264, 0x00060001);

    platform_write("\t I2C0 is initialized \n");
}

void I2C2Enable(void)
{
    WR_MEM_32(PINCNTL74, 0x00060020);
    WR_MEM_32(PINCNTL75, 0x00060020);

    WR_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL,    0x2); /*Enable I2C2 Clock*/
    while(RD_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    platform_write("\t I2C2 is initialized \n");
}

void MCA1ClkEnable(void)
{

    WR_MEM_32(CM_ALWON_MCA_1_CLKSTCTRL,    0x2); /*Enable McASP0 Clock*/
    while(RD_MEM_32(CM_ALWON_MCA_1_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    platform_write("\tMCASP1 CLOCKS init complete \n\r");
}

void BTInit(void)
{
    UWORD32 uart2_rxd;
    UWORD32 uart2_txd;
    UWORD32 uart2_cts;
    UWORD32 uart2_rts;

    platform_write("\t Running configurations for IOExpander board Bluetooth Test \n");
    GPIO0ClkEnable();
    GPIO1ClkEnable();
	GPIO2ClkEnable();

    I2C0Enable();
    I2C2Enable();

    platform_write("\t Configuring UART2 Clock to run at 48MHZ \n");

    UARTClkEnable();

    platform_write("\t Setting Up Mux Pins... \n");

    uart2_txd = RD_MEM_32(PINCNTL68);
    uart2_txd |= 0x00020002;
    WR_MEM_32(PINCNTL68, uart2_txd);  /* UART2_RXD Pulldown Rx */

    uart2_rxd = RD_MEM_32(PINCNTL69);
    uart2_rxd |= 0x00000002;
    WR_MEM_32(PINCNTL69, uart2_rxd);  /* UART2_TXD Pulldown Tx*/

    uart2_cts = RD_MEM_32(PINCNTL174);
    uart2_cts |= 0x20;
    WR_MEM_32(PINCNTL174, uart2_cts);  /* UART2_CTSn -- Input Pin - pulldown*/

    uart2_rts = RD_MEM_32(PINCNTL175);
    uart2_rts |= 0x20;
    WR_MEM_32(PINCNTL175, uart2_rts); /* UART2_RTSn -- Output Pin - pulldown*/

	WR_MEM_32(PINCNTL110, 0x20080); /* MUX SEL - GPIO1[16]*/
    WR_MEM_32(PINCNTL105, 0x20080); /* BT EN - GPIO2[5]*/

    platform_write("\t Completed configurations for IOExpander board Bluetooth Test \n");

}

void GPIO0ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_0_CLKCTRL,    0x2);
    while(RD_MEM_32(CM_ALWON_GPIO_0_CLKCTRL)!=0x2);
    platform_write("\tGPI00 CLOCKS init complete \n\r");
}

void GPIO1ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_1_CLKCTRL,    0x2);
    while(RD_MEM_32(CM_ALWON_GPIO_1_CLKCTRL)!=0x2);
    platform_write("\tGPI01 CLOCKS init complete \n\r");
}
void GPIO2ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_2_CLKCTRL,    0x2);
    while(RD_MEM_32(CM_ALWON_GPIO_2_CLKCTRL)!=0x2);
    platform_write("\tGPI02 CLOCKS init complete \n\r");
}

void UARTClkEnable(void)
{

    //Values written so UART will PASS independent of switch settings
    WR_MEM_32(PINCNTL70,0xE0001);
    WR_MEM_32(PINCNTL71,0xB0001);
    WR_MEM_32(PINCNTL72,0xE0000);
    WR_MEM_32(PINCNTL73,0xE0000);

    WR_MEM_32(CM_ALWON_L3_SLOW_CLKSTCTRL, 0x2); /*Enable the Power Domain Transition of L3 Slow Domain Peripheral*/

    WR_MEM_32(CM_ALWON_UART_0_CLKCTRL,    0x2); /*Enable UART0 Clock*/
    while(RD_MEM_32(CM_ALWON_UART_0_CLKCTRL)!=0x2);

    WR_MEM_32(CM_ALWON_UART_1_CLKCTRL,    0x2); /*Enable UART1 Clock*/
    while(RD_MEM_32(CM_ALWON_UART_1_CLKCTRL)!=0x2);

    WR_MEM_32(CM_ALWON_UART_2_CLKCTRL,    0x2); /*Enable UART2 Clock*/
    while(RD_MEM_32(CM_ALWON_UART_2_CLKCTRL)!=0x2);

    while(RD_MEM_32(CM_ALWON_L3_SLOW_CLKSTCTRL) & 0x2100 !=0x2100);
    platform_write("\tPRCM for UART0, UART1 and UART2 are Initialized\n","Output",1,1,1);
}

void DM385_System_Initialisation_TEST_device(void)
{
    ControlModule_ClkEnable();
    Unlock_PLL_Control_MMR();
    OpenFireWall();
    ALL_ADPLL_CLOCKS_ENABLE_API();
    DucatiClkEnable();
    IVAHD0ClkEnable();
    DDR3_Initialization();
    platform_write("\t DM385 System Initialization TEST Device Completed  \n","Output",1,1,1);
}

void DucatiClkEnable(void)
{

    platform_write("\tPRCM for DucatiSS is in Progress ..... \n","Output",1,1,1);

    RD_M_WR_MEM_32(RM_DEFAULT_RSTCTRL,   0x0C, 0xFFFFFFE3); /*Enable the Ducati Logic*/
    WR_MEM_32(CM_DEFAULT_DUCATI_CLKSTCTRL, 2); /*Enable Power Domain Transition*/
    WR_MEM_32(CM_DEFAULT_DUCATI_CLKCTRL,   2); /*Enable Ducati Clocks*/

    /*Check CLKIN200TR & CLKINTR  are active*/
    while(((RD_MEM_32(CM_DEFAULT_DUCATI_CLKSTCTRL)&0x300)>>8)!=3) ;
    platform_write("\tClock is Active  \n","Output",1,1,1);

    /*Write Ducate IRAM Boot Image */
    WR_MEM_32(DUCATI_BASE_ADDR,       0x10000);
    WR_MEM_32(DUCATI_BASE_ADDR+0x4,   0x9);
    WR_MEM_32(DUCATI_BASE_ADDR+0x8,   0xE7FEE7FE);
    //  platform_write("\tAfter RAM written  \n","Output",1,1,1);

    /*Large Page Translations */
    WR_MEM_32(DUCATI_MMU_CFG+0x800,         0x40000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x804,         0x80000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x808,         0xA0000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x80C,         0x60000000);

    WR_MEM_32(DUCATI_MMU_CFG+0x820,         0x40000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x824,         0x80000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x828,         0xA0000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x82C,         0x60000000);

    WR_MEM_32(DUCATI_MMU_CFG+0x840,         0x00000007);
    WR_MEM_32(DUCATI_MMU_CFG+0x844,         0x000B0007);
    WR_MEM_32(DUCATI_MMU_CFG+0x848,         0x00020007);
    WR_MEM_32(DUCATI_MMU_CFG+0x84C,         0x00020007);

    /*Medium Page*/
    WR_MEM_32(DUCATI_MMU_CFG+0x860,         0x00300000);
    WR_MEM_32(DUCATI_MMU_CFG+0x864,         0x00400000);

    WR_MEM_32(DUCATI_MMU_CFG+0x8A0,         0x40300000);
    WR_MEM_32(DUCATI_MMU_CFG+0x8A4,         0x40400000);

    WR_MEM_32(DUCATI_MMU_CFG+0x8E0,         0x00000007);
    WR_MEM_32(DUCATI_MMU_CFG+0x8E4,         0x00020007);

    /*Small Page*/
    WR_MEM_32(DUCATI_MMU_CFG+0x920,         0x00000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x924,         0x40000000);
    WR_MEM_32(DUCATI_MMU_CFG+0x928,         0x00004000);
    WR_MEM_32(DUCATI_MMU_CFG+0x92C,         0x00008000);
    WR_MEM_32(DUCATI_MMU_CFG+0x930,         0x0000C000);

    WR_MEM_32(DUCATI_MMU_CFG+0x9A0,         0x55020000);
    WR_MEM_32(DUCATI_MMU_CFG+0x9A4,         0x55080000);
    WR_MEM_32(DUCATI_MMU_CFG+0x9A8,         0x55024000);
    WR_MEM_32(DUCATI_MMU_CFG+0x9AC,         0x55028000);
    WR_MEM_32(DUCATI_MMU_CFG+0x9B0,         0x5502C000);

    WR_MEM_32(DUCATI_MMU_CFG+0xA20,         0x0001000B);
    WR_MEM_32(DUCATI_MMU_CFG+0xA24,         0x0000000B);
    WR_MEM_32(DUCATI_MMU_CFG+0xA28,         0x00010007);
    WR_MEM_32(DUCATI_MMU_CFG+0xA2C,         0x00000007);
    WR_MEM_32(DUCATI_MMU_CFG+0xA30,         0x00000007);
    platform_write("\tDUCATI MMU has been configured. User is advised to modify the mapping as needed...\n","Output",1,1,1);

    RD_M_WR_MEM_32(RM_DEFAULT_RSTCTRL,   0x00, 0xFFFFFFE3);
    /*Check for Ducati M3_0 & M3_1 out of Reset*/
    while(((RD_MEM_32(RM_DEFAULT_RSTST)&0x1C))!=0x1C) ;

    /*Check Module is in Functional Mode */
    while(((RD_MEM_32(CM_DEFAULT_DUCATI_CLKCTRL)&0x30000)>>16)!=0) ;

    platform_write("\tPRCM for DucatiSS is Done Successfully ******** \n","Output",1,1,1);
    platform_write("\tUser Can Connect to DUCATI M3s....\n","Output",1,1,1);
}

void IVAHD0ClkEnable(void)
{
    platform_write("\tPRCM for IVHD0 is in Progress, Please wait.....  \n","Output",1,1,1);
    WR_MEM_32(CM_IVAHD0_CLKSTCTRL,         2); /*Enable Power Domain Transition*/
    while(RD_MEM_32(PM_IVAHD0_PWRSTST)!=0x37);    /*Check Power is ON*/
    WR_MEM_32(CM_IVAHD0_IVAHD_CLKCTRL,     2); /*Enable IVHD0 Clocks*/
    WR_MEM_32(CM_IVAHD0_SL2_CLKCTRL,     2); /*Enable IVHD0 SL2 Clocks*/

    while(((RD_MEM_32(CM_IVAHD0_CLKSTCTRL)&0x100))!=0x100); /*IVAHD0_GCLK is Active*/

    WR_MEM_32(RM_IVAHD0_RSTCTRL,     3); /*Enable IVHD0 logic & SL2 */

    while(RD_MEM_32(RM_IVAHD0_RSTST)!=4);

    WR_MEM_32(0x58088000, 0xEAFFFFFE); /*Write Self Branch Instruction in ICONT1 ITCM 0 Location*/

    WR_MEM_32(0x58098000, 0xEAFFFFFE); /*Write Self Branch Instruction in ICONT2 ITCM 0 Location*/

    WR_MEM_32(RM_IVAHD0_RSTCTRL,     0); /*Bring ICONT1 & ICONT2 out of Reset*/

    while(RD_MEM_32(RM_IVAHD0_RSTST)!=7); /*ICONT1 & ICONT2 are out of Reset*/

    platform_write("\tPRCM for IVHD0 is Done Successfully  \n","Output",1,1,1);
    platform_write("\tUser Can Connect to ICONT1 & ICONT2 of IVHD0  \n","Output",1,1,1);
}

void EthernetInit(void)
{
	WR_MEM_32(GMII_SELECT, 0x00000000);
	GPIO0ClkEnable();
    GPIO1ClkEnable();
	GPIO2ClkEnable();
    Ethernet_PinMux_Setup();
}

void Ethernet_PinMux_Setup(void)
{
    platform_write("\t ***** Configuring ethernet Clk and Mux....***** \n");
    /*
    PINCNTL232 :- rmii_refClk            [40001]
    PINCNTL233 :- rmdio_mclk            [A0001]
    PINCNTL234 :- mdio_d                [E0001]
    */

    /* PIN MUX for EMAC0 */

    WR_MEM_32(PINCNTL232, 0x00040001);
	WR_MEM_32(PINCNTL233, 0x000E0001);
	WR_MEM_32(PINCNTL234, 0x000E0001);
	WR_MEM_32(PINCNTL235, 0x000C0001);
	WR_MEM_32(PINCNTL236, 0x000C0001);
	WR_MEM_32(PINCNTL237, 0x000C0001);
	WR_MEM_32(PINCNTL238, 0x000C0001);
	WR_MEM_32(PINCNTL239, 0x00040001);
	WR_MEM_32(PINCNTL240, 0x00040001);
	WR_MEM_32(PINCNTL241, 0x00040001);
	WR_MEM_32(PINCNTL242, 0x00040001);
	WR_MEM_32(PINCNTL243, 0x00040001);
	WR_MEM_32(PINCNTL244, 0x00040001);
	WR_MEM_32(PINCNTL245, 0x00040001);
	WR_MEM_32(PINCNTL246, 0x00040001);
    WR_MEM_32(PINCNTL247, 0x00040001);
	WR_MEM_32(PINCNTL248, 0x00040001);
	WR_MEM_32(PINCNTL249, 0x00000001);
	WR_MEM_32(PINCNTL250, 0x00000001);
	WR_MEM_32(PINCNTL251, 0x00000001);
	WR_MEM_32(PINCNTL252, 0x00000001);
	WR_MEM_32(PINCNTL253, 0x00000001);
	WR_MEM_32(PINCNTL254, 0x00000001);
	WR_MEM_32(PINCNTL255, 0x00000001);
	WR_MEM_32(PINCNTL256, 0x00000001);
	WR_MEM_32(PINCNTL257, 0x00000001);
	WR_MEM_32(PINCNTL258, 0x00000001);

    WR_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL,    0x2); /* Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    while(RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL)!=0x302);        /*Poll for Module is functional*/

    platform_write("\t ***** GMII pin mux and Clk initialized....***** \n");
}

void HDMI_Video_Init (void)
{
    unsigned int temp , temp1;
    platform_write("\tCENTAURUS ALL ADPLL INIT IS IN PROGRESS,Please wait ..... \n","Output",1,1,1);
    ALL_ADPLL_CLOCKS_ENABLE_API();
    VIDEO_0_PLL_Config ();
    DSSClkEnable();

    WR_MEM_32(0x48180324, 0x3);   //192Mhz from usb pll divide by 4 to get 48Mhz ddc_cec clock
    WR_MEM_32(0x48100100, 0x01031FFF);
    WR_MEM_32(0x48100114, 0xC010F);
    WR_MEM_32(0x48100118, 0xF);
    *(unsigned int *)0x48100000 = 0x01000000 ;     //enable dss start of frame interrupt

    VIDEO_0_PLL_Config();
    VIDEO_1_PLL_Config();
    platform_write("\t ****  CENTAURUS ALL ADPLL INIT IS COMPLETE  ******* \n","Output",1,1,1);

    hdmipinmux_cec_ddc();
    platform_write("\t ****  hdmi pin mux  complete  ******* \n","Output",1,1,1);

    temp = RD_MEM_32(0x46C00010) ;
    temp1 = ((temp & 0xFFFFFFFE)| 0x1 );
    WR_MEM_32(0x46C00300, temp1);

    temp = 0;
    while (temp ++ < 20 ) {}
    platform_write("\t ****  wrapper soft reset complete  ******* \n","Output",1,1,1);

    configure_hdmi_phy();
    platform_write("\t ****  configure hdmi phy  complete  ******* \n","Output",1,1,1);

    temp = RD_MEM_32(0x46C00070) ;
    temp1 = temp | 0x00000218;
    WR_MEM_32(0x46C00070, temp1);
    platform_write("\t ****  cec clock divider config   complete  ******* \n","Output",1,1,1);

    temp = RD_MEM_32(0x46C00044) ;
    temp1 = temp | 0x00001414;
    WR_MEM_32(0x46C00044, temp1);
    platform_write("\t ****  wrapper debounce  config   complete  ******* \n","Output",1,1,1);

    temp = RD_MEM_32(0x46C00050) ;
    temp1 = temp | 0x100;
    WR_MEM_32(0x46C00050, temp1);
    platform_write("\t ****  packing mode configuration    complete  ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00080, 0x0);
    platform_write("\t ****  disable audio    complete  ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00414, 0x1);
    WR_MEM_32(0x46C00424, 0x1);
    platform_write("\t ****  release HDMI IP CORE reset and release power down of core complete  ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00524, 0x0);
    platform_write("\t ****  video action  config of hdmi  complete  ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00420, 0x7);
    platform_write("\t ****  config input data bus width done   ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00528, 0x0);   //VID_MODE  CONFIG
    WR_MEM_32(0x46C004CC, 0x1);  //DATA ENABLE CNTRL
    WR_MEM_32(0x46C00420, 0x37);  //ENABLE VSYNC AND HSYNC
    WR_MEM_32(0x46C004F8, 0x0);  //iadjust config to enable vsync
    WR_MEM_32(0x46C00520, 0x10); // csc is bt709
    WR_MEM_32(0x46C009BC, 0x21);  //enable hdmi

    WR_MEM_32(0x46C00608, 0x20);  //tmds_ctrl
    WR_MEM_32(0x46C00904, 0x0);   //disable n/cts of actrl
    WR_MEM_32(0x46C00950, 0x0);     //disable audio
    WR_MEM_32(0x46C00414, 0x0);      //   keep audio  operation in reset state
    platform_write("\t ****  configuring AVI INFOFRAME   ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00A00 , 0x82);
    WR_MEM_32(0x46C00A04 , 0x2);
    WR_MEM_32(0x46C00A08 , 0xD);
    WR_MEM_32(0x46C00A10 , 0x1);
    WR_MEM_32(0x46C00A14 , 0xA0);
    WR_MEM_32(0x46C00A1C , 0x8F);
    platform_write("\t ****  configuring AVI INFOFRAME done   ******* \n","Output",1,1,1);

    WR_MEM_32(0x46C00538 , 0x3);    //DISABLE DEEP COLOR MODE IN DC PACKET
    WR_MEM_32(0x46C009C0 , 0x10);
    WR_MEM_32(0x46C009F8 , 0x3);      //ENABLE AND REPEAT AVI INFOFRAM TRANSMISSON
    WR_MEM_32(0x46C009FC , 0xF);    //ENABLE AND REPEAT GENEERAL PACKET TRANSMISSION
    configure_hdvenc_1080p60();
    platform_write("\t ****  configuring hdvenc for 1080p60 complete   ******* \n","Output",1,1,1);
}

void VIDEO_0_PLL_Config(void)
{
    platform_write("\t ****  DM385 VIDEO-0 ADPLL INIT IS in Progress ......... \n","Output",1,1,1);
    cmdVIDEO0PLL(CLKIN,19, 540,10);
    //platform_write("\t ****  DM385 VIDEO-0 ADPLL INIT IS Done ......... \n","Output",1,1,1);
}

void VIDEO_1_PLL_Config(void)
{
    platform_write("\t ****  DM385 VIDEO-1 ADPLL INIT IS in Progress ......... \n","Output",1,1,1);
    cmdVIDEO1PLL(CLKIN,19, 600,4);
    //platform_write("\t ****  DM385 VIDEO-1 ADPLL INIT IS Done ......... \n","Output",1,1,1);
}

void DSSClkEnable(void)
{
        *(unsigned int *)RM_DSS_RSTCTRL =0;
        platform_write("\tPower Enabled.....  \n","Output",1,1,1);
        /* Enabling DSS Clocks */
        *(unsigned int *)CM_DSS_CLKSTCTRL=2 ;
        init_delay();
        *(unsigned int *)CM_DSS_DSS_CLKCTRL=2 ;
        init_delay();
        *(unsigned int *)CM_DSS_HDMI_CLKCTRL=2 ;
        init_delay();
        platform_write("\tPRCM for DSS in Progress, Please wait.....  \n","Output",1,1,1);
        while ( (*(unsigned int *)CM_DSS_CLKSTCTRL & 0x100) != 0x100)
        {
        }
        /*Deasserting resets */
        *(unsigned int *)RM_DSS_RSTCTRL =0;
        platform_write("\tPRCM for DSS Done Successfully.....  \n","Output",1,1,1);
}

void hdmipinmux_cec_ddc(void)
{
    WR_MEM_32(0x481409B8,     0x60010);  /*hdmi_cec_mux0 pinmmr111[4] */
    WR_MEM_32(0x48140934,     0xE0002); /*hdmi_ddc_scl_mux0 pinmmr78[1]*/
    WR_MEM_32(0x48140938,     0xE0002);  /*hdmi_ddc_sda_mux0 pinmmr79[1] */
    WR_MEM_32(0x481409BC,     0x40010);  //hdmi_hpd_mux0 pinmmr112[4]
}

void configure_hdmi_phy(void)
{
    unsigned int temp , temp1;
    platform_write("\tinitialise hdmi phy  config... \n","Output",1,1,1);


     *(unsigned int *)(0x481815B0) =  0x2;
      platform_write("\t48 Mhz Clock input to HDMI ie SDIO clock output from PRCM done... \n","Output",1,1,1);

      /* Power on the phy from wrapper */
      WR_MEM_32(0x46C00040, 0x8);


       while((RD_MEM_32(0x46C00040) & 0x00000003) != 2);
    platform_write("\tHDMI PLL CONTROL MODULE IN on STAE DONE... \n","Output",1,1,1);

    WR_MEM_32(0x46C00040, 0x4A);
    while((RD_MEM_32(0x46C00040) & 0x000000FF )  != 0x5A);

    WR_MEM_32(0x46C00040, 0x8A);
    platform_write("\tinse hdmi reg  config... \n","Output",1,1,1);

    while((RD_MEM_32(0x46C00040) & 0xFF)  != 0xAA);
    platform_write("\tinitialise   config... \n","Output",1,1,1);
    platform_write("\tHDMI PHY  IN on STAE DONE... \n","Output",1,1,1);

    temp =   RD_MEM_32(0x46C00300);            /* Dummy read to PHY base to complete the SCP reset process HDMI_PHY_U_BAS*/
    temp = RD_MEM_32(0x46C00300) ;
      temp1 = ((temp & 0x3FFFFFFF)| 0x40000000 );
      WR_MEM_32(0x46C00300, temp1);

      temp = RD_MEM_32(0x46C0030C) ;
      temp1 = ((temp & 0x000FFFFF)| 0x85400000 );
    WR_MEM_32(0x46C0030C, temp1);
      WR_MEM_32(0x46C00304, 0xF0000000);
      platform_write("\tHDMI PHY  TMDS CLOCK ENABLE DONE ... \n","Output",1,1,1);

      temp = 0;
      while (temp ++ < 20 ) {}

    platform_write("\tinitialise  hdmi phy  done  \n","Output",1,1,1);
}

void configure_hdvenc_1080p60(void)
{
    WR_MEM_32(0x48106000,     0x4003A033);
    WR_MEM_32(0x48106004,     0x003F0275);
    WR_MEM_32(0x48106008,     0x1EA500BB);
    WR_MEM_32(0x4810600C,     0x1F9901C2);
    WR_MEM_32(0x48106010,     0x1FD71E67);
    WR_MEM_32(0x48106014,     0x004001C2);
    WR_MEM_32(0x48106018,     0x00200200);
    WR_MEM_32(0x4810601C,     0x1B6C0C77);
    WR_MEM_32(0x48106020,     0x1C0C0C30);
    WR_MEM_32(0x48106024,     0x1C0C0C30);
    WR_MEM_32(0x48106028,     0x84465898);
    WR_MEM_32(0x4810602C,     0x3F000028);
    WR_MEM_32(0x48106030,     0x587800BF);
    WR_MEM_32(0x48106034,     0x00000460);
    WR_MEM_32(0x48106038,     0x000C39E7);
    WR_MEM_32(0x4810603C,     0x58780118);
    WR_MEM_32(0x48106040,     0x0002A86D);
    WR_MEM_32(0x48106044,     0x00438000);
    WR_MEM_32(0x48106048,     0x05000000);
    WR_MEM_32(0x4810604C,     0x00003000);
    WR_MEM_32(0x48106050,     0x00000000);
    WR_MEM_32(0x48106054,     0x58780110);
    WR_MEM_32(0x48106058,     0x0002A86D);
    WR_MEM_32(0x4810605c,     0x00438000);
    WR_MEM_32(0x48106060,     0x05000000);
    WR_MEM_32(0x48106064,     0x00003000);
    WR_MEM_32(0x48106068,     0x00000000);
    WR_MEM_32(0x4810606c,     0x00000000);

}

void LEDInit(void)
{
 /* GP3[3] configuration  */
 WR_MEM_32(PINCNTL211, 0x00040080);

 /* GP0[9] configuration  */
 WR_MEM_32(PINCNTL16,  0x000C0080);

 /* GP1[24] configuration  */
 WR_MEM_32(PINCNTL123, 0x00060080);

 /* GP1[25] configuration  */
 WR_MEM_32(PINCNTL124, 0x00060080);

 /* GP1[30] configuration  */
 WR_MEM_32(PINCNTL132, 0x00040080);

 GPIO0ClkEnable();
 GPIO1ClkEnable();
}

void SDMMCInit(void)
{
    platform_write("\t ***** SD MMC Setup....***** \n");
    ALL_ADPLL_CLOCKS_ENABLE_API();
    WR_MEM_32(PINCNTL1, 0x00050001);
    WR_MEM_32(PINCNTL2, 0x00060001);
    WR_MEM_32(PINCNTL3, 0x00060001);
    WR_MEM_32(PINCNTL4, 0x00060001);
    WR_MEM_32(PINCNTL5, 0x00060001);
    WR_MEM_32(PINCNTL6, 0x00060001);
	WR_MEM_32(PINCNTL80, 0x00060002);
    MMC1ClkEnable();
    platform_write("\t ***** MMC/SD is initialized....***** \n");

}

void MMC0ClkEnable(void)
{
    WR_MEM_32(CM_ALWON_MMC_HS_0_CLKSTCTRL,    0x2); /* SD0 Functional Clock Enable */
    while(RD_MEM_32(CM_ALWON_MMC_HS_0_CLKSTCTRL)!=0x2);    /*Poll for Module is functional*/
    platform_write("\tMMC0 CLOCKS init complete \n\r");
}

void MMC1ClkEnable(void)
{
    WR_MEM_32(CM_ALWON_MMC_HS_1_CLKSTCTRL,    0x2); /*Enable UART0 Clock*/
    while(RD_MEM_32(CM_ALWON_MMC_HS_1_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/
    platform_write("\tMMC1 CLOCKS init complete \n\r");

}

void SwitchInit(void)
{
	WR_MEM_32(PINCNTL14, 0x00060004);  /* MCA0- AHCLKX */
    WR_MEM_32(PINCNTL17, 0x00060001);  /* MCA0- ACLKX */
    WR_MEM_32(PINCNTL18, 0x00060001);  /* MCA0- AFSX */
    WR_MEM_32(PINCNTL19, 0x00060001);  /* MCA0-ACLKR */
    WR_MEM_32(PINCNTL20, 0x000C0001);  /* MCA0-AFSR */

    WR_MEM_32(PINCNTL22, 0x00060001);  /* MCA0- AXR1 */
    WR_MEM_32(PINCNTL23, 0x00060001);  /* MCA0- AXR2 */

    MCA0ClkEnable();
    platform_write("\t ***** AIC3104 McASP0 is initialized....***** \n");

	GPIO0ClkEnable();
    GPIO1ClkEnable();
	GPIO2ClkEnable();
	WR_MEM_32(PINCNTL41, 0x000E0080);
	WR_MEM_32(PINCNTL42, 0x000E0080);
	WR_MEM_32(PINCNTL232,0x00040080);
	platform_write("\t ***** GPIO Clocks Enabled and Pinmux done....***** \n");
}

void MCA0ClkEnable(void)
{

    WR_MEM_32(CM_ALWON_MCA_0_CLKSTCTRL,    0x2); /*Enable McASP0 Clock*/
    while(RD_MEM_32(CM_ALWON_MCA_0_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    platform_write("\tMCASP0 CLOCKS init complete \n\r");
}

void USBInit(void)
{

    platform_write("\t ***** USB on Baseboard Initialization ....***** \n");

    ALL_ADPLL_CLOCKS_ENABLE_API();
    /* USB_PINMUX_Config */
    WR_MEM_32(PINCNTL270, 0x00060001);
    WR_MEM_32(PINCNTL14, 0x00060080);

    WR_MEM_32(0x48140620, 0x3C1E64F4); /* PHY INIT */
    WR_MEM_32(0x48140628, 0x3C1E64F4); /* PHY INIT */

    USBClkEnable();
    platform_write("\t*** USB Pin Muxing and enabling the clock Done  ***\n ","Output",1,1,1);
}

void USBClkEnable(void)
{
    WR_MEM_32(CM_ALWON2_USB_CLKCTRL,    0x2); /*Enable USB Clock*/
   //while((RD_MEM_32(CM_ALWON2_USB_CLKCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(RM_ALWON2_RSTCTRL, 0x9F); /* Disable local reset */
    //while((RD_MEM_32(RM_ALWON2_RSTST) & 0x60) != 0x60);        /*Poll for Module is functional*/
    //WR_MEM_32(RM_ALWON2_RSTST, 0x60); /* Write 1 to Clear */

    platform_write("\t*** USB Clock Enable completed  *** ","Output",1,1,1);
}

void DM388_WLANInit(void)
{
    platform_write("\t Running configurations for WLAN Test \n");
    ALL_ADPLL_CLOCKS_ENABLE_API();
	GPIO1ClkEnable();
    GPIO2ClkEnable();
    MMC0ClkEnable();
    /* SD/SDIO Pins */
    WR_MEM_32(PINCNTL8,  0x00060001);
    WR_MEM_32(PINCNTL9,  0x000E0001);
    WR_MEM_32(PINCNTL10, 0x000E0001);
    WR_MEM_32(PINCNTL11, 0x000E0001);
    WR_MEM_32(PINCNTL12, 0x000E0001);
    WR_MEM_32(PINCNTL13, 0x000E0001);

    WR_MEM_32(PINCNTL106, 0x00040080); /* WLAN EN GPIO*/

    platform_write("\t Completed configurations for WLAN Test \n");

}

/** **************************************************************************
 * \n \brief Routine to initialize the DM388 EVM board
 *
 * This routine initializes the DM388 EVM Base Board
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
INT32 DM388_Init (void)
{
	//platform_write ("Test Suite version number is %d.%d.\r\n", MAJOR_VERSION, MINOR_VERSION);
	//platform_write ("Build Date =  %s : Time = %s.\r\n", __DATE__, __TIME__);
	//ControlModule_ClkEnable();
	//Unlock_PLL_Control_MMR();
	//OpenFireWall();
	ALL_ADPLL_CLOCKS_ENABLE_API();
	DDR3_Initialization();

	return (SUCCESS);
}

/** **************************************************************************
 * \n \brief Delay loop used for Micro-second delay
 *
 * Delay loop used for Micro-second delay
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
void DM388_wait
(
	UINT32 u32Delay
)
{
    volatile UINT32 u32LoopCntr;
    for ( u32LoopCntr = 0 ; u32LoopCntr < u32Delay ; u32LoopCntr++ ){ };
}


/** **************************************************************************
 * \n \brief Micro-second delay routine
 *
 * Utility routine for micro-seconds
 *
 * \param  u32usec	[IN] 	Delay in micro-seconds
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
void DM388_usecDelay
(
	UINT32 u32usec
)
{
    #ifdef ARM_SIDE
        DM388_wait( u32usec * 2 );
    #elif DSP_SIDE
        DM388_wait( u32usec * 28 );
    #endif
    DM388_wait( u32usec * 2 );
}
